/*
 * Copyright (c) 2022 Jim Seymour (jseymour+sshguard@LinxNet.com)
 *
 * attack_parser_re.h
 *
 * Check a log line for an attack signature
 */
#ifndef _ATTACK_PARSER_RE_H
#define _ATTACK_PARSER_RE_H

#ifndef USE_NATIVE_PCRE
#ifndef USE_PCRE
#include <regex.h>
#else
#include <pcreposix.h>
#endif
#define atre_regex_t regex_t
#else
#include <pcre.h>

#ifndef REG_NOMATCH
#define REG_NOMATCH 1
#endif

typedef struct atre_regex_t {
    pcre *re;
    int startoffset;
    pcre_extra *sd;
    int cap_cnt;
} atre_regex_t;
#endif

#ifndef ATRE_CONFIG_FILE_PATH
#if defined(USE_PCRE) || defined(USE_NATIVE_PCRE)
#define ATRE_CONFIG_FILE_PATH "/etc/sshguard/attack_parser_re.pcre"
#else
#define ATRE_CONFIG_FILE_PATH "/etc/sshguard/attack_parser_re.conf"
#endif
#endif

#define attack_parser_re_fin attack_parser_re_cleaner

/* Attack parser regex linked list entry */
typedef struct atre_list_t {
    struct  atre_list_t *next;
    struct  atre_list_t *prev;		/* allows us to remove defective config lines */
    atre_regex_t atre_pat;
    int dangerousness;
    int faulty_re_warnings;		/* to limit warnings about possible faulty regexps */
    int line_nbr;
} atre_list_t;

/*
 * For non-USE_NATIVE_PCRE builds, misconfigured regexps with too few or
 * too many captures can't be detected until runtime. Too many captures
 * can only result from a faulty regexp.  We can safely prune such regexp
 * list entries.  But too few could result from a wonky logfile line, in
 * which case removing the regexp responsible could result in pruning a
 * valid regexp entry.  But if too few IS the result of a faulty regexp:
 * We don't want to hammer the logfile with repeated warning messages.
 *
 * Thus:
 */
#define MAX_FAULTY_RE_WARNINGS 100

/* attack_parser_re_init return values */
#define ATRE_CONF_SUCCESS 0	/* successful (re-)config */
#define ATRE_CONF_NOCONF  1	/* no (re-)config */
#define ATRE_CONF_ERR    -1	/* fatal (re-)configuration error */

extern int attack_parser_re_init(const char *cfigPath, atre_list_t **ret_atre_list_p, const int debugging, const int logging_verbosity, const int use_syslog);
extern int parse_line_re(char *log_line, attack_t *attack);
extern void attack_parser_re_cleaner(atre_list_t **atre_list_pp);
extern void reload_attack_parser_re_conf(void);
extern enum service svc_name_to_code(const char *svc_name);
extern const char *atre_service_to_name(enum service code);
extern void atre_log(int priority, const char *fmt, ...);
extern const char *svc_name_code_iterator(int * const indx, int * const svc_code);

#endif /* attack_parser_re.h */
